/*
 * Top-level routines for querying data from Myricom switches
 */
#ifndef _WIN32
#include <unistd.h>
#endif

#include "libfma.h"
#include "lf_internal.h"
#include "lf_fabric.h"
#include "lf_switch.h"

/* local prototypes */
static int lf_get_switch_data_protocol(lf_enclosure_t *ep);
static int lf_query_switch_type(struct lf_enclosure *ep);
static void lf_update_linecard_data(struct lf_linecard *lp);
static void lf_update_switch_data(struct lf_enclosure *ep);
static int lf_check_for_missing_switch_data(struct lf_enclosure *ep);
static void flush_switch_data_changes(struct lf_enclosure *ep);


/*
 * Query data from an enclosure.  The enclosure data strect may be specified
 * or NULL.  If NULL, we first go query the switch to see what type it is and
 * what's in it and allocate the appropriate structures.  The structure we
 * return can be re-used in subsequent calls to lf_query_switch.
 */
int
lf_query_switch(
  char *name,
  struct lf_enclosure **epp)
{
  struct lf_enclosure *ep;
  struct lf_enclosure_data *edp;
  int tries;
  int rc;

  /* If not called with an enclosure data struct, allocate one and
   * get initial enclosure data filled in.
   */
  ep = *epp;
  if (ep == NULL) {
    LF_CALLOC(ep, struct lf_enclosure, 1);
    LF_DUP_STRING(ep->name, name);
    LF_CALLOC(ep->data, struct lf_enclosure_data, 1);
  }

  /* If type not known yet, discover it */
  if (ep->product_id == NULL || strlen(ep->product_id) == 0) {
    rc = lf_query_switch_type(ep);
    if (rc != 0) {
      LF_ERROR(("Error probing for switch type for %s", name));
    }
    sleep(2);		/* some switches don't like quick re-connects */
  }

  /*
   * If we don't know the protocol to talk yet, figure that out now
   */
  edp = ep->data;
  if (edp->data_proto == LF_SWITCH_PROTO_UNKNOWN) {
    if (lf_get_switch_data_protocol(ep) == -1) {
      LF_ERROR(("Getting protocol for switch %d", ep->name));
    }
  }

  /* flush any un-read changes */
  flush_switch_data_changes(ep);

  /*
   * Perform the actual query of the switch
   */
  tries = 0;
  do {
    switch (edp->data_proto) {
      case LF_SWITCH_PROTO_TELNET_1:
	rc = lf_query_switch_telnet_1(ep);
	break;
   
      case LF_SWITCH_PROTO_X16_HTML:
	rc = lf_query_switch_x16_html(ep);
	break;
   
      case LF_SWITCH_PROTO_X32_HTML:
	rc = lf_query_switch_x32_html(ep);
	break;
   
      default:
	LF_ERROR(("Bad switch protocol for %s = %d", name, edp->data_proto));
	break;
    }

    /* see if anything is missing */
    if (rc == LF_QUERY_OK) {
      rc = lf_check_for_missing_switch_data(ep);
      if (rc == LF_QUERY_FATAL) {
	LF_ERROR(("Error checking for missing linecards"));
      }
    }

  } while (rc == LF_QUERY_RETRY && tries < LF_SWITCH_QUERY_RETRY_COUNT);

  if (rc != LF_QUERY_OK) LF_ERROR(("Error querying switch %s", name));

  /* copy the data just read into current area */
  lf_update_switch_data(ep);

  /* if all ok, return enclosure ptr */
  if (*epp == NULL) *epp = ep;

  return 0;

 except:
  return -1;
}
 
/*
 * Loop through polling the different switch types to see what this is
 */
static int
lf_query_switch_type(
  struct lf_enclosure *ep)
{
  int rc;

#if 0		/* enable for 10G support */
  /* See if the switch responds to telnet_1 protocol */
  rc = lf_probe_switch_telnet_1(ep);
#else
  rc = -1;
#endif

  /* check for an xbar-16 style switch */
  if (rc != 0) {
    rc = lf_probe_switch_x16(ep);
  }

  /* check for an xbar-32 style switch */
  if (rc != 0) {
    rc = lf_probe_switch_x32(ep);
  }

  /* if we got a hit, fill in internal links */
  if (rc == 0) {
    lf_create_implicit_links(ep);
  }

  return rc;
}

/*
 * Determine the protocol this enclosure speaks
 */
static int
lf_get_switch_data_protocol(
  struct lf_enclosure *ep)
{
  struct lf_enclosure_data *edp;

  edp = ep->data;
  if (strcmp(ep->product_id, "M3-E16") == 0) {
    edp->data_proto = LF_SWITCH_PROTO_X16_HTML;

  } else if (strcmp(ep->product_id, "M3-E32") == 0) {
    edp->data_proto = LF_SWITCH_PROTO_X16_HTML;

  } else if (strcmp(ep->product_id, "M3-E64") == 0) {
    edp->data_proto = LF_SWITCH_PROTO_X16_HTML;

  } else if (strcmp(ep->product_id, "M3-E128") == 0) {
    edp->data_proto = LF_SWITCH_PROTO_X16_HTML;

  } else if (strcmp(ep->product_id, "M3-CLOS-ENCL") == 0) {
    edp->data_proto = LF_SWITCH_PROTO_X32_HTML;

  } else if (strcmp(ep->product_id, "M3-SPINE-ENCL") == 0) {
    edp->data_proto = LF_SWITCH_PROTO_X32_HTML;

  } else if (strcmp(ep->product_id, "10G-CLOS-ENCL") == 0) {
    edp->data_proto = LF_SWITCH_PROTO_TELNET_1;

  } else {
    return -1;
  }

  return 0;
}

/*
 * copy data for linecard from cur to old and new to cur
 */
static void
lf_update_linecard_data(
  struct lf_linecard *lp)
{
  int x;
  int p;
  
  /* update the linecard data */
  lp->data->old_vals = lp->data->cur_vals;
  lp->data->cur_vals = lp->data->new_vals;

  for (x=0; x<lp->num_xbars; ++x) {
    struct lf_xbar *xp;

    xp = LF_XBAR(lp->xbars[x]);

    xp->xbar_data->old_vals = xp->xbar_data->cur_vals;
    xp->xbar_data->cur_vals = xp->xbar_data->new_vals;

    for (p=0; p<xp->num_ports; ++p) {
      xp->data[p].old_vals = xp->data[p].cur_vals;
      xp->data[p].cur_vals = xp->data[p].new_vals;
    }
  }
 
  for (p=0; p<lp->num_xcvrs; ++p) {
    struct lf_xcvr *xcp;
    xcp = LF_XCVR(lp->xcvrs[p]);
    xcp->data->old_vals = xcp->data->cur_vals;
    xcp->data->cur_vals = xcp->data->new_vals;
  }

  /* record as unseen until next pass */
  lp->data->seen = FALSE;
  lp->data->change_cnt = 0;
}

/*
 * copy data for an enclosure from cur to old and new to cur
 */
static void
lf_update_switch_data(
  lf_enclosure_t *ep)
{
  int s;

  for (s=0; s<ep->num_slots; ++s) {
    struct lf_linecard *lp;
    lp = ep->slots[s];
    if (lp != NULL) {
      lf_update_linecard_data(lp);
    }
  }
}

/*
 * Check that all elements reported in properly
 */
static int
lf_check_for_missing_switch_data(
  struct lf_enclosure *ep)
{
  struct lf_linecard *lp;
  int s;
  int rc;

  rc = LF_QUERY_OK;

  for (s=0; s<ep->num_slots; ++s) {
    lp = ep->slots[s];
    if (lp == NULL) continue;

    if (!lp->data->seen) {

      /* allow x16 switches to have cards pop in and out of existance */
      if (ep->data->data_proto == LF_SWITCH_PROTO_X16_HTML) {
	++lp->data->not_seen_cnt;
	if (lp->data->not_seen_cnt >= 3) {
	  rc = lf_note_missing_linecard(ep, s);
	  if (rc == -1) goto except;
	} else {
	  rc = LF_QUERY_RETRY;
	}
      }
    } else {
      lp->data->not_seen_cnt = 0;
    }
  }
  return rc;

 except:
  return LF_QUERY_FATAL;
}

/*
 * Add a change record reflecting a missing linecard
 */
int
lf_note_missing_linecard(
  struct lf_enclosure *ep,
  int slot)
{
  struct lf_switch_data_change *cp;

  /* allocate a change record */
  LF_CALLOC(cp, struct lf_switch_data_change, 1);

  /* fill it in */
  cp->dc_type = LF_SWITCH_CHANGE_MISSING_LINECARD;
  cp->c.missing_linecard.slot = slot;

  /* and link it into the list */
  cp->next = ep->data->change_list;
  ep->data->change_list = cp;

  return 0;

 except:
  return -1;
}

/*
 * Add a change record reflecting a new or changed linecard
 */
int
lf_note_new_linecard(
  struct lf_enclosure *ep,
  int slot,
  char *product_id,
  char *serial_no)
{
  struct lf_switch_data_change *cp;

  /* allocate a change record */
  LF_CALLOC(cp, struct lf_switch_data_change, 1);

#if 0
printf("new/change LC, s/n=%s, id=%s\n", serial_no, product_id);
if (ep->slots[slot] != NULL) printf("  was s/n=%s, id=%s\n", ep->slots[slot]->serial_no, ep->slots[slot]->product_id);
#endif

  /* fill it in */
  cp->dc_type = (ep->slots[slot] == NULL) ?
    		LF_SWITCH_CHANGE_NEW_LINECARD :
    		LF_SWITCH_CHANGE_CHANGED_LINECARD;
  cp->c.new_linecard.slot = slot;
  LF_DUP_STRING(cp->c.new_linecard.product_id, product_id);
  LF_DUP_STRING(cp->c.new_linecard.serial_no, serial_no);

  /* and link it into the list */
  cp->next = ep->data->change_list;
  ep->data->change_list = cp;

  return 0;

 except:
  return -1;
}

/*
 * Free a switch change report
 */
void
lf_free_switch_change(
  struct lf_switch_data_change *cp)
{
  switch (cp->dc_type) {
    case LF_SWITCH_CHANGE_NEW_LINECARD:
    case LF_SWITCH_CHANGE_CHANGED_LINECARD:
      LF_FREE(cp->c.new_linecard.product_id);
      LF_FREE(cp->c.new_linecard.serial_no);
      break;
  }

  LF_FREE(cp);
}

/*
 * Free all changes on an enclosure
 */
static void
flush_switch_data_changes(
  struct lf_enclosure *ep)
{
  struct lf_switch_data_change *cp;
  struct lf_switch_data_change *ncp;

  cp = ep->data->change_list;
  ep->data->change_list = NULL;
  
  while (cp != NULL) {
    ncp = cp->next;
    lf_free_switch_change(cp);
    cp = ncp;
  }
}
